package com.appodeal.consent

import android.app.Activity
import androidx.annotation.UiThread

/**
 * Interface for handling consent forms.
 * Implement this interface to show a consent form and listen for events related to its lifecycle.
 */
interface ConsentForm {
    /**
     * Shows the consent form.
     *
     * @param activity [Activity] The context in which the consent form will be displayed.
     * @param listener The listener [OnConsentFormDismissedListener] to be notified when the consent
     * form is dismissed. Can be null.
     */
    @UiThread
    fun show(activity: Activity, listener: OnConsentFormDismissedListener?)
}

/**
 * Functional interface for handling successful loading of a consent form.
 * Implement this interface to receive callbacks when a consent form is successfully loaded.
 */
fun interface OnConsentFormLoadSuccessListener {
    /**
     * Called when a consent form is successfully loaded.
     *
     * @param consentForm The loaded consent form [ConsentForm] instance.
     */
    fun onConsentFormLoadSuccess(consentForm: ConsentForm)
}

/**
 * Functional interface for handling failure to load a consent form.
 * Implement this interface to receive callbacks when loading a consent form fails.
 */
fun interface OnConsentFormLoadFailureListener {
    /**
     * Called when loading a consent form fails.
     *
     * @param error The specific error encountered during the consent form loading process.
     * Possible values are defined in the [ConsentManagerError] enum.
     */
    fun onConsentFormLoadFailure(error: ConsentManagerError)
}

/**
 * Functional interface for handling the dismissal of a consent form.
 * Implement this interface to receive callbacks when a consent form is dismissed.
 */
fun interface OnConsentFormDismissedListener {
    /**
     * Called when a consent form is dismissed.
     *
     * @param error The specific error encountered during the consent form dismissal, if any.
     * Can be null if the dismissal was successful or one of the [ConsentManagerError] enum values.
     */
    fun onConsentFormDismissed(error: ConsentManagerError?)
}
