package com.appodeal.consent

import com.google.android.ump.ConsentInformation as GoogleUmpConsentInformation

/**
 * Sealed class representing different types of consent information.
 *
 * @param type The type of consent information (e.g., UMP or NONE).
 */
sealed class ConsentInformation(val type: CmpType) {
    abstract val status: ConsentStatus

    /**
     * Class representing User Messaging Platform (UMP) consent information.
     *
     * @property information The Google UMP consent information.
     */
    class UmpConsentInformation(
        private val information: GoogleUmpConsentInformation
    ) : ConsentInformation(CmpType.Ump) {
        override val status: ConsentStatus get() = information.consentStatus.toConsentStatus()

        override fun toString(): String = "UmpConsentInformation(status=$status, type=$type)"

        /**
         * Converts Google UMP consent status to Appodeal ConsentStatus.
         *
         * @return The corresponding ConsentStatus.
         */
        private fun Int.toConsentStatus(): ConsentStatus {
            return when (this) {
                GoogleUmpConsentInformation.ConsentStatus.UNKNOWN -> ConsentStatus.Unknown
                GoogleUmpConsentInformation.ConsentStatus.REQUIRED -> ConsentStatus.Required
                GoogleUmpConsentInformation.ConsentStatus.NOT_REQUIRED -> ConsentStatus.NotRequired
                GoogleUmpConsentInformation.ConsentStatus.OBTAINED -> ConsentStatus.Obtained
                else -> ConsentStatus.Unknown
            }
        }
    }

    /**
     * Class representing no specific consent information.
     *
     * @param status The status of consent information.
     */
    class NoneConsentInformation(
        override val status: ConsentStatus,
    ) : ConsentInformation(CmpType.None) {
        override fun toString(): String = "NoneConsentInformation(status=$status, type=$type)"
    }
}
