package com.appodeal.consent

import android.app.Activity
import android.content.Context
import com.appodeal.consent.ConsentManagerError.FormPresentationNotRequired
import com.appodeal.consent.cache.PrivacyPreferences
import com.appodeal.consent.form.GetConsentFormUseCase
import com.appodeal.consent.networking.LoadConsentInfoUseCase
import com.appodeal.consent.revoke.RevokeConsentUseCase
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch

/**
 * Singleton object for managing consent in the Appodeal Consent Manager.
 * Use this object to update consent information, load consent forms, and check whether ads can be shown based on consent status.
 */
object ConsentManager {

    /**
     * Coroutine scope for executing operations on the main thread.
     */
    private val uiScope: CoroutineScope by lazy { CoroutineScope(Dispatchers.Main) }

    /**
     * Privacy preferences instance for storing and retrieving consent-related data.
     */
    private val privacyPreferences: PrivacyPreferences by lazy { PrivacyPreferences() }

    /**
     * Use case for loading consent information.
     */
    private val loadConsentInfo: LoadConsentInfoUseCase get() = LoadConsentInfoUseCase()

    /**
     * Use case for getting consent forms.
     */
    private val getConsentForm: GetConsentFormUseCase get() = GetConsentFormUseCase()

    /**
     * Use case for revoking consent.
     */
    private val revokeConsent: RevokeConsentUseCase get() = RevokeConsentUseCase()

    /**
     * Gets the current information about the user's consent status, as obtained from the last consent update.
     *
     * @return The [ConsentInformation] containing the user's consent status information.
     *         Returns null if no information is available or if consent information has not been updated.
     */
    internal var consentInformation: ConsentInformation? = null

    /**
     * The current consent status.
     */
    @JvmStatic
    val status: ConsentStatus get() = consentInformation?.status ?: ConsentStatus.Unknown

    /**
     * Checks whether ads can be shown based on the current consent status.
     *
     * @return True if ads can be shown, false otherwise.
     */
    @JvmStatic
    fun canShowAds(): Boolean {
        return status == ConsentStatus.NotRequired || status == ConsentStatus.Obtained
    }

    /**
     * Requests an update to the consent information.
     *
     * @param parameters [ConsentUpdateRequestParameters] The parameters for the consent update request.
     * @param callback [ConsentInfoUpdateCallback] The callback to be notified of the update result.
     */
    @JvmStatic
    fun requestConsentInfoUpdate(
        parameters: ConsentUpdateRequestParameters,
        callback: ConsentInfoUpdateCallback,
    ) {
        uiScope.launch {
            loadConsentInfo(
                parameters = parameters,
                privacyPreferences = privacyPreferences,
            ).onFailure { error ->
                callback.onFailed(error.asConsentManagerError())
            }.onSuccess { consentInformation ->
                this@ConsentManager.consentInformation = consentInformation
                callback.onUpdated()
            }
        }
    }

    /**
     * Loads a consent form.
     *
     * @param context [Context] The context in which the consent form is loaded.
     * @param successListener [OnConsentFormLoadSuccessListener] The listener to be notified when the consent form is successfully loaded.
     * @param failureListener [OnConsentFormLoadFailureListener] The listener to be notified when loading the consent form fails.
     */
    @JvmStatic
    fun load(
        context: Context,
        successListener: OnConsentFormLoadSuccessListener,
        failureListener: OnConsentFormLoadFailureListener,
    ) {
        uiScope.launch {
            getConsentForm(
                context = context,
                privacyPreferences = privacyPreferences,
                consentInformation = consentInformation,
            ).onFailure { error ->
                failureListener.onConsentFormLoadFailure(error.asConsentManagerError())
            }.onSuccess { form ->
                successListener.onConsentFormLoadSuccess(form)
            }
        }
    }

    /**
     * Loads and shows a consent form if required.
     *
     * @param activity [Activity] The activity in which the consent form is loaded and shown.
     * @param dismissedListener [OnConsentFormDismissedListener] The listener to be notified when the consent form is dismissed.
     */
    @JvmStatic
    fun loadAndShowConsentFormIfRequired(
        activity: Activity,
        dismissedListener: OnConsentFormDismissedListener,
    ) {
        load(
            context = activity.applicationContext,
            successListener = { form ->
                if (status == ConsentStatus.Required) {
                    form.show(activity, dismissedListener)
                } else {
                    dismissedListener.onConsentFormDismissed(FormPresentationNotRequired)
                }
            },
            failureListener = { error ->
                dismissedListener.onConsentFormDismissed(error)
            }
        )
    }

    /**
     * Revokes consent, resetting the consent status to unknown.
     *
     * @param context [Context] The context in which consent is revoked.
     */
    @JvmStatic
    fun revoke(context: Context) {
        uiScope.launch {
            revokeConsent(
                context = context,
                privacyPreferences = privacyPreferences,
                consentInformation = consentInformation,
            )
            consentInformation = null
        }
    }
}
