package com.appodeal.consent

/**
 * Represents errors that can occur in the Appodeal Consent Manager.
 * This sealed class extends Throwable, and specific error instances are defined as objects within the class.
 */
sealed class ConsentManagerError(
    override val message: String,
    override val cause: Throwable? = null
) : Throwable(message, cause) {

    /**
     * Represents a timeout error during consent management.
     */
    class TimeoutError(message: String) : ConsentManagerError(message)

    /**
     * Represents a request error during consent management.
     */
    class RequestError(message: String? = null, cause: Throwable? = null) : ConsentManagerError(
        message = "Request error: ${message ?: cause?.message}",
        cause = cause
    )

    /**
     * Represents a server error during consent management.
     */
    class ServerError(message: String) : ConsentManagerError(message)

    /**
     * Represents an error related to form caching during consent management.
     */
    class FormCacheError(message: String) : ConsentManagerError(message)

    /**
     * Represents an error when attempting to show a form that is not ready.
     */
    class FormNotReadyError(message: String) : ConsentManagerError("Form not ready error: $message")

    /**
     * Represents an error when activity is destroyed.
     */
    class ActivityIsDestroyedError(message: String) : ConsentManagerError(message)

    /**
     * Represents an error when attempting to show a form that has already been shown.
     */
    object FormAlreadyShown : ConsentManagerError("Form is already shown")

    /**
     * Represents an error when presenting a form is not required.
     */
    object FormPresentationNotRequired : ConsentManagerError("Presentation not required")

    /**
     * Represents a general internal error during consent management.
     */
    class InternalError(message: String? = null, cause: Throwable? = null) : ConsentManagerError(
        message = "Internal error: ${message ?: cause?.message}",
        cause = cause
    )
}

/**
 * Extension function to convert a Throwable into a ConsentManagerError.
 */
internal fun Throwable.asConsentManagerError(): ConsentManagerError {
    this.printStackTrace()
    return when (this) {
        // Consent Manager errors
        is ConsentManagerError -> this
        // Other errors
        else -> ConsentManagerError.InternalError(message = this.message, cause = this)
    }
}
