package com.appodeal.consent

/**
 * Enum class representing the possible consent statuses in the Appodeal Consent Manager.
 * Each enum value corresponds to a different consent status, and a companion object provides a utility
 * function to obtain a ConsentStatus from a given status string.
 */
enum class ConsentStatus(val statusName: String) {
    /**
     * Represents an unknown consent status.
     */
    Unknown("UNKNOWN"),

    /**
     * Represents a required consent status.
     */
    Required("REQUIRED"),

    /**
     * Represents a not required consent status.
     */
    NotRequired("NOT_REQUIRED"),

    /**
     * Represents an obtained consent status.
     */
    Obtained("OBTAINED");

    /**
     * Companion object providing utility functions for the ConsentStatus enum class.
     */
    companion object {
        /**
         * Gets the ConsentStatus corresponding to the provided status string.
         *
         * @param status The status string to match against enum values.
         * @return The ConsentStatus enum value matching the provided status string,
         *         or ConsentStatus.Unknown if no match is found.
         */
        fun get(status: String?) = values().firstOrNull { it.statusName == status } ?: Unknown
    }
}
