package com.appodeal.ads

import android.app.Activity
import android.content.Context
import com.appodeal.ads.inapp.AppodealPurchaseListener
import com.appodeal.ads.inapp.InAppPurchase
import com.appodeal.ads.inapp.InAppPurchaseValidateCallback
import com.appodeal.ads.initializing.ApdInitializationCallback
import com.appodeal.ads.revenue.AdRevenueCallbacks
import com.appodeal.ads.rewarded.Reward
import com.appodeal.ads.segments.SegmentConstants
import com.appodeal.ads.utils.Log
import java.util.*

object Appodeal {
    const val NONE = 0 // 0000000000
    const val INTERSTITIAL = 3 // 0000000011
    const val BANNER = 4 // 0000000100
    const val BANNER_BOTTOM = 8 // 0000001000
    const val BANNER_TOP = 16 // 0000010000
    const val BANNER_LEFT = 1024
    const val BANNER_RIGHT = 2048
    const val BANNER_VIEW = 64 // 0001000000
    const val REWARDED_VIDEO = 128 // 0010000000
    const val MREC = 256 // 0100000000
    const val NATIVE = 512 // 1000000000

    internal const val ALL = 4095 // 1111111111

    /**
     * Initializes the Appodeal SDK.
     *
     * To initialize only interstitials use `Appodeal.initialize(this, appKey, Appodeal.INTERSTITIAL);`
     *
     * To initialize only banners use `Appodeal.initialize(this, appKey, Appodeal.BANNER);`
     *
     * To initialize only rewarded video use `Appodeal.initialize(this, appKey, Appodeal.REWARDED_VIDEO);`
     *
     * To initialize only 300*250 banners use `Appodeal.initialize(this, appKey, Appodeal.MREC);`
     *
     * To initialize only native ads use `Appodeal.initialize(this, appKey, Appodeal.NATIVE);`
     *
     * @param context  context of your application. [Context]
     * @param appKey   Appodeal app key you received when you created an app.
     * @param adTypes  type of advertising you want to initialize.
     * @param callback (optional) Appodeal initialization callback [ApdInitializationCallback]
     */
    @JvmStatic
    @JvmOverloads
    fun initialize(
        context: Context,
        appKey: String,
        adTypes: Int,
        callback: ApdInitializationCallback? = null,
    ) {
        SdkCore.initialize(context, appKey, adTypes, callback)
    }

    /**
     * Set self-hosted Bidon environment endpoint
     *
     *  @param endpoint Bidon environment endpoint
     */
    @JvmStatic
    fun setBidonEndpoint(endpoint: String) {
        SdkCore.setBidonEndpoint(endpoint)
    }

    /**
     * Get self-hosted Bidon environment endpoint
     *
     * @return Bidon environment endpoint
     */
    @JvmStatic
    fun getBidonEndpoint(): String? {
        return SdkCore.getBidonEndpoint()
    }

    /**
     * Check if ad type is initialized
     *
     * @param adType type of advertising
     * @return `true` if ad type is initialized
     */
    @JvmStatic
    fun isInitialized(adType: Int): Boolean {
        return SdkCore.isInitialized(adType)
    }

    /**
     * Check if auto cache enabled for  this ad type
     * AutoCache enabled by default for all ad types
     *
     * @param adType type of advertising
     * @return `true` if auto cache enabled for this ad type
     */
    @JvmStatic
    fun isAutoCacheEnabled(adType: Int): Boolean {
        return SdkCore.isAutoCacheEnabled(adType)
    }

    /**
     * Set Request callbacks
     *
     * @param callbacks implementation of [com.appodeal.ads.AppodealRequestCallbacks]
     */
    @JvmStatic
    @Deprecated(
        "As of Appodeal SDK version 3.2.0, will be removed in a future release.",
        ReplaceWith("Appodeal.setAdRevenueCallbacks(callbacks)"),
    )
    fun setRequestCallbacks(callbacks: AppodealRequestCallbacks?) {
        SdkCore.setRequestCallbacks(callbacks)
    }

    /**
     * Set Ad revenue callbacks
     *
     * @param callbacks implementation of [com.appodeal.ads.revenue.AdRevenueCallbacks]
     */
    @JvmStatic
    fun setAdRevenueCallbacks(callbacks: AdRevenueCallbacks?) {
        SdkCore.adRevenueCallbacks = callbacks
    }

    /**
     * Set Interstitial ads callbacks
     *
     * @param callbacks implementation of [com.appodeal.ads.InterstitialCallbacks]
     */
    @JvmStatic
    fun setInterstitialCallbacks(callbacks: InterstitialCallbacks?) {
        SdkCore.setInterstitialCallbacks(callbacks)
    }

    /**
     * Set Rewarded video ads callbacks
     *
     * @param callbacks implementation of [com.appodeal.ads.RewardedVideoCallbacks]
     */
    @JvmStatic
    fun setRewardedVideoCallbacks(callbacks: RewardedVideoCallbacks?) {
        SdkCore.setRewardedVideoCallbacks(callbacks)
    }

    /**
     * Set Banner ads callbacks
     *
     * @param callbacks implementation of [com.appodeal.ads.BannerCallbacks]
     */
    @JvmStatic
    fun setBannerCallbacks(callbacks: BannerCallbacks?) {
        SdkCore.setBannerCallbacks(callbacks)
    }

    /**
     * Set MREC ads callbacks
     *
     * @param callbacks implementation of [com.appodeal.ads.MrecCallbacks]
     */
    @JvmStatic
    fun setMrecCallbacks(callbacks: MrecCallbacks?) {
        SdkCore.setMrecCallbacks(callbacks)
    }

    /**
     * Set Native ads callbacks
     *
     * @param callbacks implementation of [com.appodeal.ads.NativeCallbacks]
     */
    @JvmStatic
    fun setNativeCallbacks(callbacks: NativeCallbacks?) {
        SdkCore.setNativeCallbacks(callbacks)
    }

    /**
     * Set preferred type of the requested Native ads
     *
     * @param contentType type of Native ads [com.appodeal.ads.NativeMediaViewContentType]
     */
    @JvmStatic
    fun setPreferredNativeContentType(contentType: NativeMediaViewContentType) {
        SdkCore.nativeMediaViewContent = contentType
    }

    /**
     * @return preferred type of the requested Native ads [com.appodeal.ads.NativeMediaViewContentType]
     */
    @JvmStatic
    fun getPreferredNativeContentType(): NativeMediaViewContentType {
        return SdkCore.nativeMediaViewContent
    }

    /**
     * Start caching specified count of ads.
     *
     * @param activity activity of your application. [android.app.Activity]
     * @param adTypes  type of advertising you want to cache.
     * @param count    (optional) the numbers of native ads to cache.
     */
    @JvmStatic
    @JvmOverloads
    fun cache(activity: Activity, adTypes: Int, count: Int = 1) {
        SdkCore.cache(activity, adTypes, count)
    }

    /**
     * Show advertising.
     *
     * @param activity      activity of your application. [android.app.Activity]
     * @param adTypes       type of advertising you want to show.
     * @param placementName (optional) name of placement.
     * @return `true` if ads shown
     */
    @JvmStatic
    @JvmOverloads
    fun show(activity: Activity, adTypes: Int, placementName: String = SegmentConstants.DEFAULT): Boolean {
        return SdkCore.show(activity, adTypes, placementName)
    }

    /**
     * Hide banner or MREC
     *
     * @param activity activity of your application. [android.app.Activity]
     * @param adTypes  Appodeal.BANNER or Appodeal.MREC
     */
    @JvmStatic
    fun hide(activity: Activity, adTypes: Int) {
        SdkCore.hide(activity, adTypes)
    }

    /**
     * Start or stop auto caching new ads when current ads was shown.
     * AutoCache enabled by default for all ad types
     *
     * @param adTypes   type of advertising.
     * @param autoCache true to use auto cache, false to not.
     */
    @JvmStatic
    fun setAutoCache(adTypes: Int, autoCache: Boolean) {
        SdkCore.setAutoCache(adTypes, autoCache)
    }

    /**
     * Triggering onLoaded callback when precache loaded
     *
     * @param adTypes                   type of advertising. Currently supported for interstitials,
     * rewarded video, banners and MREC.
     * @param triggerOnLoadedOnPrecache true - onLoaded will trigger when precache or normal ad were loaded.
     * false - onLoaded will trigger only when normal ad was loaded.
     * triggerOnLoadedOnPrecache is false by default
     */
    @JvmStatic
    fun setTriggerOnLoadedOnPrecache(
        adTypes: Int,
        triggerOnLoadedOnPrecache: Boolean
    ) {
        SdkCore.setTriggerOnLoadedOnPrecache(adTypes, triggerOnLoadedOnPrecache)
    }

    /**
     * Checking if shared ads instance across activities is enabled.
     * sharedAdsInstanceAcrossActivities is false by default.
     *
     * @return `true` if shared ads instance across activities is enabled, otherwise - `false`
     */
    @JvmStatic
    @Deprecated("As of Appodeal SDK version 3.1, will be changed in a future release.")
    fun isSharedAdsInstanceAcrossActivities(): Boolean {
        return SdkCore.isSharedAdsInstanceAcrossActivities
    }

    /**
     * Enabling shared ads instance across activities
     *
     * @param sharedAdsInstanceAcrossActivities enabling or disabling shared ads instance across activities.
     *                                          sharedAdsInstanceAcrossActivities is false by default.
     */
    @JvmStatic
    @Deprecated("As of Appodeal SDK version 3.1, will be changed in a future release.")
    fun setSharedAdsInstanceAcrossActivities(sharedAdsInstanceAcrossActivities: Boolean) {
        SdkCore.isSharedAdsInstanceAcrossActivities = sharedAdsInstanceAcrossActivities
    }

    /**
     * Checking if ad is loaded.
     *
     * To check if interstitial is loaded use `Appodeal.isLoaded(Appodeal.INTERSTITIAL);`
     *
     * To check if rewarded video is loaded use `Appodeal.isLoaded(Appodeal.REWARDED_VIDEO);`
     *
     * To check if banner is loaded use `Appodeal.isLoaded(Appodeal.BANNER);`
     *
     * To check if 300*250 banner is loaded use `Appodeal.isLoaded(Appodeal.MREC);`
     *
     * To check if native ad is loaded use `Appodeal.isLoaded(Appodeal.NATIVE);`
     *
     * @param adTypes type of advertising.
     * @return `true` if ads currently loaded and can be shown.
     */
    @JvmStatic
    fun isLoaded(adTypes: Int): Boolean {
        return SdkCore.isLoaded(adTypes)
    }

    /**
     * Checking if loaded ad is precache.
     *
     * @param adType type of advertising. Currently supported only for interstitials.
     * @return `true` if currently loaded ads is precache.
     */
    @JvmStatic
    fun isPrecache(adType: Int): Boolean {
        return SdkCore.isPrecache(adType)
    }

    /**
     * Checking if loaded ad is precache and can shown for specific placement.
     *
     * @param adType type of advertising. Currently supported only for interstitials.
     * @param placementName placement name. Using "default" by default.
     * @return `true` if currently loaded ads is precache and ad can be shown with this placement.
     */
    @JvmStatic
    @JvmOverloads
    fun isPrecacheByPlacement(adType: Int, placementName: String = SegmentConstants.DEFAULT): Boolean {
        return SdkCore.isPrecacheByPlacement(adType, placementName)
    }

    /**
     * Set Banner view id.
     * Now you can show banner in the view specified: `Appodeal.show(this, Appodeal.BANNER_VIEW);`
     *
     * @param bannerViewId id of banner view.
     */
    @JvmStatic
    fun setBannerViewId(bannerViewId: Int) {
        SdkCore.setBannerViewId(bannerViewId)
    }

    /**
     * Using programmatically created banner view.
     *
     * @param context context of your application. [android.content.Context]
     * @return programmatically created banner view.
     */
    @JvmStatic
    fun getBannerView(context: Context): BannerView {
        return SdkCore.getBannerView(context)
    }

    /**
     * Enabling or disabling smart banners.
     * Smart banners are enabled by default.
     *
     * @param enabled enabling or disabling loading smart banners.
     */
    @JvmStatic
    fun setSmartBanners(enabled: Boolean) {
        SdkCore.setSmartBanners(enabled)
    }

    /**
     * Checking if smart banners are enabled.
     * Smart banners are enabled by default.
     *
     * @return `true` if smart banners are enabled, otherwise - `false`
     */
    @JvmStatic
    fun isSmartBannersEnabled(): Boolean {
        return SdkCore.isSmartBannersEnabled
    }

    /**
     * Enabling or disabling 728*90 banners.
     * Banners 728*90 are disabled by default.
     *
     * @param enabled enabling or disabling loading 728*90 banners.
     */
    @JvmStatic
    fun set728x90Banners(enabled: Boolean) {
        SdkCore.set728x90Banners(enabled)
    }

    /**
     * Enabling animation of banners.
     * Banner animation is enabled by default.
     *
     * @param animate enabling or disabling animations.
     */
    @JvmStatic
    fun setBannerAnimation(animate: Boolean) {
        SdkCore.setBannerAnimation(animate)
    }

    /**
     * Setting banners inverse rotation (by default: left = -90, right = 90).
     *
     * @param leftBannerRotation  rotation for [Appodeal.BANNER_LEFT]
     * @param rightBannerRotation rotation for [Appodeal.BANNER_LEFT]
     */
    @JvmStatic
    fun setBannerRotation(leftBannerRotation: Int, rightBannerRotation: Int) {
        SdkCore.setBannerRotation(leftBannerRotation, rightBannerRotation)
    }

    /**
     * Set using of safe area status.
     * Use safe area is enabled by default.
     *
     * @param useSafeArea enabling or disabling using of safe area.
     */
    @JvmStatic
    fun setUseSafeArea(useSafeArea: Boolean) {
        AppodealSettings.isUseSafeArea = useSafeArea
    }

    /**
     * Set MREC view id.
     * Now you can show MREC in the view specified: `Appodeal.show(this, Appodeal.MREC);`
     *
     * @param mrecViewId id of banner view.
     */
    @JvmStatic
    fun setMrecViewId(mrecViewId: Int) {
        SdkCore.setMrecViewId(mrecViewId)
    }

    /**
     * Using programmatically created MREC view.
     *
     * Add view to your layout and than show banner in it: `Appodeal.show(this, Appodeal.MREC);`
     *
     * @param context context of your application. [android.app.Activity]
     * @return programmatically created MREC view.
     */
    @JvmStatic
    fun getMrecView(context: Context): MrecView {
        return SdkCore.getMrecView(context)
    }

    /**
     * Tracks in-app purchase information and sends info to our servers for analytics.
     *
     * @param context  context of your activity. [android.content.Context]
     * @param amount   amount of purchase.
     * @param currency currency of purchase.
     */
    @JvmStatic
    fun trackInAppPurchase(
        context: Context,
        amount: Double,
        currency: String,
    ) {
        SdkCore.trackInAppPurchase(context, amount, currency)
    }

    /**
     * Getting list of ad networks for certain ad type.
     *
     * Add view to your layout and than show banner in it: `Appodeal.show(this, Appodeal.MREC);`
     *
     * @param adTypes (optional) type of advertising.
     * @return List of networks name.
     */
    @JvmStatic
    @JvmOverloads
    fun getNetworks(adTypes: Int = ALL): List<String> {
        return ArrayList(SdkCore.getNetworks(adTypes))
    }

    /**
     * Disabling specified network for specified ad types.
     *
     * @param network network name, see [com.appodeal.ads.adnetworks.MediationAdNetwork]
     * @param adTypes (optional) type of advertising.
     */
    @JvmStatic
    @JvmOverloads
    fun disableNetwork(network: String, adTypes: Int = ALL) {
        SdkCore.disableNetwork(network, adTypes)
    }

    /**
     * Set user id
     *
     * @param userId - user id
     */
    @JvmStatic
    fun setUserId(userId: String) {
        SdkCore.setUserId(userId)
    }

    /**
     * Get user id
     *
     * @return user id
     */
    @JvmStatic
    fun getUserId(): String? {
        return SdkCore.userId
    }

    /**
     * Get SDK version.
     *
     * @return SDK version.
     */
    @JvmStatic
    fun getVersion(): String = SdkCore.version

    /**
     * Get framework name.
     *
     * @return framework name.
     */
    @JvmStatic
    fun getFrameworkName(): String? = SdkCore.frameworkName

    /**
     * Get plugin version.
     *
     * @return plugin version.
     */
    @JvmStatic
    fun getPluginVersion(): String? = SdkCore.pluginVersion

    /**
     * Get engine version.
     *
     * @return engine version.
     */
    @JvmStatic
    fun getEngineVersion(): String? = SdkCore.engineVersion

    /**
     * Get matched segment Id.
     *
     * @return matched segment Id.
     */
    @JvmStatic
    fun getSegmentId(): Long = SdkCore.segmentId

    /**
     * Get SDK build date.
     *
     * @return SDK build date.
     */
    @JvmStatic
    fun getBuildDate(): Date = SdkCore.buildDate

    /**
     * Enabling test mode.
     * In test mode test ads will be shown and debug data will be written to logcat.
     *
     * @param testMode true to enable test mode.
     */
    @JvmStatic
    fun setTesting(testMode: Boolean) {
        SdkCore.setTesting(testMode)
    }

    /**
     * Get current log level
     *
     * @return logLevel log level [com.appodeal.ads.utils.Log.LogLevel].
     */
    @JvmStatic
    fun getLogLevel(): Log.LogLevel = SdkCore.logLevel

    /**
     * Set log level.
     * All logs will be written with tag "Appodeal".
     *
     * @param logLevel log level [com.appodeal.ads.utils.Log.LogLevel].
     */
    @JvmStatic
    fun setLogLevel(logLevel: Log.LogLevel) {
        SdkCore.logLevel = logLevel
    }

    /**
     * Set custom segment filter
     *
     * @param name  name of the filter
     * @param value filter value
     */
    @JvmStatic
    fun setCustomFilter(name: String, value: Boolean) {
        SdkCore.setCustomFilter(name, value as Any)
    }

    /**
     * Set custom segment filter
     *
     * @param name  name of the filter
     * @param value filter value
     */
    @JvmStatic
    fun setCustomFilter(name: String, value: Int) {
        SdkCore.setCustomFilter(name, value.toFloat() as Any)
    }

    /**
     * Set custom segment filter
     *
     * @param name  name of the filter
     * @param value filter value
     */
    @JvmStatic
    fun setCustomFilter(name: String, value: Double) {
        SdkCore.setCustomFilter(name, value.toFloat() as Any)
    }

    /**
     * Set custom segment filter
     *
     * @param name  name of the filter
     * @param value filter value
     */
    @JvmStatic
    fun setCustomFilter(name: String, value: String) {
        SdkCore.setCustomFilter(name, value as Any)
    }

    /**
     * Set custom segment filter
     *
     * @param name  name of the filter
     * @param value filter value
     */
    @JvmStatic
    fun setCustomFilter(name: String, value: Any?) {
        SdkCore.setCustomFilter(name, value)
    }

    /**
     * Get list of loaded native ads
     *
     * @param count count of ads
     * @return list of Native ads [com.appodeal.ads.NativeAd]
     */
    @JvmStatic
    fun getNativeAds(count: Int): List<NativeAd> {
        return ArrayList(SdkCore.getNativeAds(count))
    }

    /**
     * Get count of available Native ads
     *
     * @return count of available Native ads [com.appodeal.ads.NativeAd]
     */
    @JvmStatic
    fun getAvailableNativeAdsCount(): Int = SdkCore.availableNativeAdsCount

    /**
     * Check if ad with specific ad type can be shown with placement
     *
     * @param adTypes       type of advertising
     * @param placementName (optional) name of placement
     * @return `true` if ad can be shown with this placement, `false` if can't
     */
    @JvmStatic
    @JvmOverloads
    fun canShow(adTypes: Int, placementName: String = SegmentConstants.DEFAULT): Boolean {
        return SdkCore.canShow(adTypes, placementName)
    }

    /**
     * Get reward parameters for placement
     *
     * @param placementName (optional) name of placement
     * @return reward object with amount and reward currency
     */
    @JvmStatic
    @JvmOverloads
    fun getReward(placementName: String = SegmentConstants.DEFAULT): Reward {
        return SdkCore.getReward(placementName)
    }

    @JvmStatic
    @JvmOverloads
    fun setFramework(frameworkName: String?, pluginVersion: String?, engineVersion: String? = null) {
        SdkCore.setFramework(frameworkName, pluginVersion, engineVersion)
    }

    /**
     * Mute video if calls muted on device
     *
     * @param muteVideosIfCallsMuted true - mute videos if call volume is 0.
     * muteVideosIfCallsMuted is false by default
     */
    @JvmStatic
    fun muteVideosIfCallsMuted(muteVideosIfCallsMuted: Boolean) {
        SdkCore.muteVideosIfCallsMuted(muteVideosIfCallsMuted)
    }

    /**
     * Start TestActivity to test integration
     *
     * @param activity Activity of your application. [android.app.Activity]
     */
    @JvmStatic
    fun startTestActivity(activity: Activity) {
        SdkCore.startTestActivity(activity)
    }

    /**
     * Disables data collection for kids apps. You must enable it for US citizens under the age of 13 only.
     *
     * @param value true to disable data collection for kids apps
     * @see [COPPA Law](https://www.ftc.gov/legal-library/browse/rules/childrens-online-privacy-protection-rule-coppa)
     */
    @JvmStatic
    fun setChildDirectedTreatment(value: Boolean?) {
        SdkCore.setChildDirectedTreatment(value)
    }

    /**
     * Destroy cached ad.
     *
     * @param adType type of advertising
     */
    @JvmStatic
    fun destroy(adTypes: Int) {
        SdkCore.destroy(adTypes)
    }

    /**
     * Add extra data to Appodeal
     *
     * @param key   associated with value
     * @param value which will be saved in extra data by key
     */
    @JvmStatic
    fun setExtraData(key: String, value: String) {
        SdkCore.setExtraData(key, value)
    }

    /**
     * Add extra data to Appodeal
     *
     * @param key   associated with value
     * @param value which will be saved in extra data by key
     */
    @JvmStatic
    fun setExtraData(key: String, value: Int) {
        SdkCore.setExtraData(key, value)
    }

    /**
     * Add extra data to Appodeal
     *
     * @param key   associated with value
     * @param value which will be saved in extra data by key
     */
    @JvmStatic
    fun setExtraData(key: String, value: Double) {
        SdkCore.setExtraData(key, value)
    }

    /**
     * Add extra data to Appodeal
     *
     * @param key   associated with value
     * @param value which will be saved in extra data by key
     */
    @JvmStatic
    fun setExtraData(key: String, value: Boolean) {
        SdkCore.setExtraData(key, value)
    }

    /**
     * Add extra data to Appodeal
     *
     * @param key   associated with value
     * @param value which will be saved in extra data by key
     */
    @JvmStatic
    fun setExtraData(key: String, value: Any?) {
        SdkCore.setExtraData(key, value)
    }

    /**
     * Get predicted ecpm for certain ad type
     *
     * @param adType type of advertising
     * @return the predicted eCPM from the next block in the caching queue
     */
    @JvmStatic
    fun getPredictedEcpm(adType: Int): Double {
        return SdkCore.getPredictedEcpm(adType)
    }

    /**
     * Get predicted ecpm for certain ad type
     *
     * @param adType type of advertising
     * @param placementName placement name. Using "default" by default.
     * @return the predicted eCPM from the next block in the caching queue and ad can be shown with this placement.
     */
    @JvmStatic
    @JvmOverloads
    fun getPredictedEcpmByPlacement(adType: Int, placementName: String = SegmentConstants.DEFAULT): Double {
        return SdkCore.getPredictedEcpmByPlacement(adType, placementName)
    }

    /**
     * Logging event in one of connected attribution service
     *
     * @param eventName event name
     * @param params    params of event
     * @param service   service to which parameters should be sent, see [com.appodeal.ads.AppodealServices].
     */
    @JvmStatic
    @JvmOverloads
    fun logEvent(eventName: String, params: Map<String, Any?>?, service: Int = AppodealServices.ALL) {
        SdkCore.logEvent(eventName, params, service)
    }

    /**
     * Validate in-app purchase in one of connected attribution service
     *
     * @param context  context of your activity. [android.content.Context]
     * @param purchase purchase object. [InAppPurchase]
     * @param callback callback that returns the result of the in-app purchase validation. [InAppPurchaseValidateCallback]
     */
    @JvmStatic
    fun validateInAppPurchase(
        context: Context,
        purchase: InAppPurchase,
        callback: InAppPurchaseValidateCallback?,
    ) {
        SdkCore.validateInAppPurchase(context, purchase, callback)
    }

    /**
     * Subscribe for automatically purchase validation and tracking
     *
     * @param purchaseListener listener that returns the result [AppodealPurchaseListener]
     */
    @JvmStatic
    fun setPurchaseListener(purchaseListener: AppodealPurchaseListener?) {
        SdkCore.setPurchaseListener(purchaseListener)
    }
}
