package com.appodeal.ads.nativead

import android.content.Context
import android.util.AttributeSet
import android.view.View
import android.view.ViewGroup
import android.widget.TextView
import androidx.annotation.ColorInt
import com.appodeal.ads.Events
import com.appodeal.ads.NativeAd
import com.appodeal.ads.R.styleable.NativeAdView
import com.appodeal.ads.R.styleable.NativeAdView_adAttributionViewId
import com.appodeal.ads.R.styleable.NativeAdView_adChoicePosition
import com.appodeal.ads.R.styleable.NativeAdView_callToActionViewId
import com.appodeal.ads.R.styleable.NativeAdView_descriptionViewId
import com.appodeal.ads.R.styleable.NativeAdView_iconViewId
import com.appodeal.ads.R.styleable.NativeAdView_mediaViewId
import com.appodeal.ads.R.styleable.NativeAdView_ratingViewId
import com.appodeal.ads.R.styleable.NativeAdView_titleViewId
import com.appodeal.ads.modules.common.internal.LogConstants.EVENT_ASSETS_ERROR
import com.appodeal.ads.nativead.Position.END_BOTTOM
import com.appodeal.ads.nativead.Position.END_TOP
import com.appodeal.ads.nativead.Position.START_BOTTOM
import com.appodeal.ads.nativead.Position.START_TOP
import com.appodeal.ads.segments.SegmentConstants
import com.appodeal.ads.utils.Log

/**
 * [NativeAdView] - custom root view for showing [NativeAd].
 * @property [titleView] Required view. Title of the native ad. Maximum 25 symbols of the titleView should always be displayed. You can add ellipsis at the end if the titleView is longer.
 * @property [callToActionView] Required view. Call-to-action view. Should be displayed without truncation on a visible button.
 * @property [descriptionView] Optional view. Text descriptionView of the native ad. If you choose to display the descriptionView, you should display maximum 100 characters. You can add ellipsis at the end.
 * @property [ratingView] Optional view. Rating of the app in [0-5] range.
 * @property [adAttributionView] Required view. Advertising Indicator. This is a TextView labeled "Ad". You can specify the color of the text and background.
 * @property [adChoiceView] Special ad icon provided by ad network. It add automatically and you don't need to attach it to [NativeAdView] by yourself.
 * @property [iconView] Optional view(if you use [NativeMediaView]). Icon view of NativeAd.
 * @property [mediaView] Optional view(if you use [NativeIconView]). Media view of NativeAd.
 *
 * The [NativeAdView] must necessarily contain at least one of the following elements: [NativeMediaView]/[NativeIconView]
 *
 * @see <a href="https://docs.appodeal.com/android/ad-types/native">Documentation</a>
 * */
open class NativeAdView @JvmOverloads constructor(
    context: Context,
    attrs: AttributeSet? = null,
    defStyleAttr: Int = 0,
) : NativeAdViewContainer(context, attrs, defStyleAttr) {
    private var currentAd: NativeAd? = null

    private var titleViewId = NO_ID
    private var callToActionViewId = NO_ID
    private var descriptionViewId = NO_ID
    private var ratingViewId = NO_ID
    private var iconViewId = NO_ID
    private var mediaViewId = NO_ID
    private var adAttributionViewId = NO_ID

    var titleView: View? = null
    var callToActionView: View? = null
    var descriptionView: View? = null
    var ratingView: View? = null
    var iconView: NativeIconView? = null
    var mediaView: NativeMediaView? = null
    var adAttributionView: TextView? = null
    var adChoiceView: ViewGroup? = null

    val clickableViews: List<View>
        get() = addClickableViews()

    init {
        var adChoicePosition = END_BOTTOM.ordinal
        if (attrs != null) {
            val attrsArray = context.obtainStyledAttributes(attrs, NativeAdView, defStyleAttr, 0)
            with(attrsArray) {
                try {
                    if (!isInEditMode) visibility = View.GONE
                    titleViewId = getResourceId(NativeAdView_titleViewId, NO_ID)
                    callToActionViewId = getResourceId(NativeAdView_callToActionViewId, NO_ID)
                    descriptionViewId = getResourceId(NativeAdView_descriptionViewId, NO_ID)
                    ratingViewId = getResourceId(NativeAdView_ratingViewId, NO_ID)
                    iconViewId = getResourceId(NativeAdView_iconViewId, NO_ID)
                    mediaViewId = getResourceId(NativeAdView_mediaViewId, NO_ID)
                    adAttributionViewId = getResourceId(NativeAdView_adAttributionViewId, NO_ID)
                    adChoicePosition = getInt(NativeAdView_adChoicePosition, adChoicePosition)
                } finally {
                    recycle()
                }
            }
        }
        addAdChoiceView(adChoicePosition)
    }

    override fun onFinishInflate() {
        super.onFinishInflate()
        if (titleViewId != NO_ID && titleView == null) {
            titleView = findViewById(titleViewId)
        }
        if (callToActionViewId != NO_ID && callToActionView == null) {
            callToActionView = findViewById(callToActionViewId)
        }
        if (descriptionViewId != NO_ID && descriptionView == null) {
            descriptionView = findViewById(descriptionViewId)
        }
        if (ratingViewId != NO_ID && ratingView == null) {
            ratingView = findViewById(ratingViewId)
        }
        if (iconViewId != NO_ID && iconView == null) {
            iconView = findViewById(iconViewId)
        }
        if (mediaViewId != NO_ID && mediaView == null) {
            mediaView = findViewById(mediaViewId)
        }
        if (adAttributionViewId != NO_ID && adAttributionView == null) {
            adAttributionView = findViewById(adAttributionViewId)
        }
        configureEditModePreview()
    }

    open fun isViewValid(): Boolean {
        Events.NativeAdView.isViewValid.track()
        return when {
            titleView == null -> {
                Log.log(EVENT_ASSETS_ERROR, "NativeAdView is invalid", "titleView was not provided")
                false
            }

            callToActionView == null -> {
                Log.log(
                    EVENT_ASSETS_ERROR,
                    "NativeAdView is invalid",
                    "callToActionView was not provided"
                )
                false
            }

            adAttributionView == null -> {
                Log.log(
                    EVENT_ASSETS_ERROR,
                    "NativeAdView is invalid",
                    "adAttributionView was not provided"
                )
                false
            }

            iconView == null && mediaView == null -> {
                Log.log(
                    EVENT_ASSETS_ERROR,
                    "NativeAdView is invalid",
                    "NativeAdView must contain a NativeMediaView or NativeIconView"
                )
                false
            }

            else -> true
        }
    }

    /**
     * Register the view for displaying a native ad.
     * @param nativeAd The [NativeAd] to be displayed.
     * @param placementName The placement name for displaying the ad. Default is "default".
     * @return [Boolean] true if the view is successfully registered, false otherwise.
     */
    @JvmOverloads
    fun registerView(
        nativeAd: NativeAd,
        placementName: String = SegmentConstants.DEFAULT,
    ): Boolean {
        Events.NativeAdView.registerView.track()
        if (isViewValid().not()) return false
        unregisterPrevious(currentAd = currentAd, newAd = nativeAd)
        currentAd = nativeAd
        if (nativeAd.canShow(context, placementName).not()) return false
        visibility = VISIBLE
        (nativeAd as NativeAdViewRegister).registerView(this, placementName)
        return true
    }

    /**
     * Unregister the view from displaying the currently registered native ad.
     */
    fun unregisterView() {
        Events.NativeAdView.unregisterView.track()
        (currentAd as? NativeAdViewRegister)?.unregisterView()
    }

    /**
     * Destroy the native ad view and perform any necessary cleanup.
     * This method should be called when the native ad is no longer needed.
     */
    fun destroy() {
        Events.NativeAdView.destroy.track()
        unregisterView()
        visibility = GONE
        currentAd?.destroy()
        currentAd = null
        titleView = null
        callToActionView = null
        descriptionView = null
        ratingView = null
        iconView = null
        mediaView = null
        adChoiceView = null
        adAttributionView = null
    }

    /**
     * Set text color for [adAttributionView].
     *
     * @param color should be ColorInt. See [android.graphics.Color].
     * */
    fun setAdAttributionTextColor(@ColorInt color: Int) {
        if (color.isColor()) {
            adAttributionView?.setTextColor(color)
        } else {
            Log.log(
                EVENT_ASSETS_ERROR,
                "setAdAttributionTextColor",
                "provided color should be ColorInt. Use android.graphics.Color"
            )
        }
    }

    /**
     * Set background color for [adAttributionView].
     *
     * @param color should be ColorInt. See [android.graphics.Color].
     * */
    fun setAdAttributionBackground(@ColorInt color: Int) {
        if (color.isColor()) {
            adAttributionView?.setBackgroundColor(color)
        } else {
            Log.log(
                EVENT_ASSETS_ERROR,
                "setAdAttributionBackground",
                "provided color should be ColorInt. Use android.graphics.Color"
            )
        }
    }

    /**
     * Set position for [adChoiceView].
     * @param position - you can specify one of 4 [Position]:
     * [START_TOP], [START_BOTTOM], [END_TOP], [END_BOTTOM]
     * */
    fun setAdChoicesPosition(position: Position) {
        (adChoiceView?.layoutParams as? LayoutParams?)?.gravity = position.gravity
    }

    private fun unregisterPrevious(currentAd: NativeAd?, newAd: NativeAd) {
        (currentAd as? NativeAdViewRegister)?.unregisterView()
        (newAd as? NativeAdViewRegister)?.unregisterView()
    }

    private fun addClickableViews(): List<View> = buildList {
        titleView?.let { add(it) }
        descriptionView?.let { add(it) }
        callToActionView?.let { add(it) }
        iconView?.let { add(it) }
        mediaView?.let { add(it) }
        ratingView?.let { add(it) }
    }
}
