package com.appodeal.ads

import android.graphics.drawable.Drawable
import android.net.Uri
import com.appodeal.ads.ImageData.LocalUri
import com.appodeal.ads.VideoData.Remote

/**
 * [MediaAssets] is an object that contains media content to be set in the [com.appodeal.ads.nativead.NativeAdView].
 * @property [icon] - content for [com.appodeal.ads.nativead.NativeIconView]
 * @property [mainImage] - static content for [com.appodeal.ads.nativead.NativeMediaView]
 * @property [video] - video content for [com.appodeal.ads.nativead.NativeMediaView]
 *
 * All [MediaAssets] are downloaded and set to [com.appodeal.ads.nativead.NativeIconView]/[com.appodeal.ads.nativead.NativeMediaView] automatically in the Appodeal SDK side.
 * You don't need to implement such functionality.
 * It may useful for debugging only.
 * */
class MediaAssets(
    var icon: ImageData? = ImageData.Autoload,
    var mainImage: ImageData? = ImageData.Autoload,
    var video: VideoData? = VideoData.Autoload,
)

/**
 * [ImageData] is an wrapper of static image content to be set in [com.appodeal.ads.nativead.NativeIconView] and [com.appodeal.ads.nativead.NativeMediaView].
 *
 * @types:
 * [Autoload] - marker indicating that AdNetwork will download content automatically.
 * [Remote] - marker indicating that content should be load manually. Contains [Remote.remoteUrl] for loading [ImageData].
 * By the time [NativeAd] is displayed in [com.appodeal.ads.nativead.NativeAdView], [ImageData] cannot be marked as [Remote]
 * [LocalUri] - marker indicating that content was loaded. Contains [LocalUri.localUri] to receiving [ImageData]
 * [LocalDrawable] - marker indicating that content was loaded. Contains [LocalDrawable.drawable] to setting to [com.appodeal.ads.nativead.NativeIconView] or [com.appodeal.ads.nativead.NativeMediaView].
 *
 * All [ImageData] are downloaded and set to [com.appodeal.ads.nativead.NativeIconView]/[com.appodeal.ads.nativead.NativeMediaView] automatically in the Appodeal SDK side.
 * You do not need to implement such functionality.
 * */
sealed interface ImageData {
    object Autoload : ImageData
    data class Remote(val remoteUrl: String) : ImageData
    class LocalUri(val localUri: Uri) : ImageData
    class LocalDrawable(val drawable: Drawable) : ImageData
}

/**
 * [VideoData] is an wrapper of video content to be set in [com.appodeal.ads.nativead.NativeMediaView].
 *
 * @types:
 * [Autoload] - marker indicating that AdNetwork will download content automatically.
 * [Remote] - marker indicating that content should be load manually. Contains [Remote.remoteUrl] for loading [VideoData].
 * [LocalUri] - marker indicating that content was loaded. Contains [LocalUri.localUri] to receiving [ImageData]
 *
 * [VideoData] is downloaded and set to [com.appodeal.ads.nativead.NativeMediaView] automatically in the Appodeal SDK side.
 * You do not need to implement such functionality.
 * */
sealed interface VideoData {
    object Autoload : VideoData
    data class Remote(val remoteUrl: String) : VideoData
    class LocalUri(val localUri: Uri) : VideoData
}
