package com.appodeal.ads.inapp

/**
 * In-app purchase object
 * @property type type of in app (purchase or subscription). {@link com.appodeal.ads.inapp.InAppPurchase.Type} Usage: Adjust/AppsFlyer.
 * @property publicKey public key from Google Developer Console. Usage: AppsFlyer. @see <a href=""https://support.google.com/googleplay/android-developer/answer/186113">Link</a>
 * @property signature transaction signature (returned from Google API when the purchase is completed). Usage: Adjust/AppsFlyer.
 * @property purchaseData product purchased in JSON format (returned from Google API when the purchase is completed). Usage: AppsFlyer.
 * @property developerPayload Product purchased developer payload (returned from Google API when the purchase is completed). Usage: Adjust.
 * @property price purchase revenue. Usage: Adjust/AppsFlyer.
 * @property currency Purchase currency. Usage: Adjust/AppsFlyer.
 * @property sku Stock keeping unit id. Usage for Adjust.
 * @property orderId product purchased unique order id for the transaction (returned from Google API when the purchase is completed). Usage: Adjust.
 * @property purchaseToken product purchased token (returned from Google API when the purchase is completed). Usage for Adjust.
 * @property purchaseTimestamp product purchased timestamp (returned from Google API when the purchase is completed). Usage: Adjust.
 * @property additionalParameters additional parameters of the purchase event. Usage: Adjust/AppsFlyer.
 * */
class InAppPurchase private constructor(
    val type: Type,
    val publicKey: String?,
    val signature: String?,
    val purchaseData: String?,
    val developerPayload: String?,
    val price: String?,
    val currency: String?,
    val sku: String?,
    val orderId: String?,
    val purchaseToken: String?,
    val purchaseTimestamp: Long = 0,
    val additionalParameters: Map<String, String> = emptyMap()
) {

    override fun toString(): String {
        return "price='$price', currency='$currency'"
    }

    /**
     * Type of in app (purchase or subscription). Usage: Adjust/AppsFlyer.
     * */
    enum class Type {
        InApp, Subs
    }

    /**
     * Class-builder for [InAppPurchase].
     * */
    class Builder @JvmOverloads constructor(
        var type: Type,
        var publicKey: String? = null,
        var signature: String? = null,
        var purchaseData: String? = null,
        var developerPayload: String? = null,
        var price: String? = null,
        var currency: String? = null,
        var sku: String? = null,
        var orderId: String? = null,
        var purchaseToken: String? = null,
        var purchaseTimestamp: Long = 0,
        var additionalParameters: Map<String, String> = emptyMap()
    ) {
        /**
         * @param publicKey public key from Google Developer Console. Usage: AppsFlyer. @see <a href=""https://support.google.com/googleplay/android-developer/answer/186113">Link</a>
         * */
        fun withPublicKey(publicKey: String?) = apply { this.publicKey = publicKey }

        /**
         * @param signature transaction signature (returned from Google API when the purchase is completed). Usage: Adjust/AppsFlyer.
         * */
        fun withSignature(signature: String?) = apply { this.signature = signature }

        /**
         * @param purchaseData product purchased in JSON format (returned from Google API when the purchase is completed). Usage: AppsFlyer.
         * */
        fun withPurchaseData(purchaseData: String?) = apply { this.purchaseData = purchaseData }

        /**
         * @param price purchase revenue. Usage: Adjust/AppsFlyer.
         * */
        fun withPrice(price: String?) = apply { this.price = price }

        /**
         * @param currency Purchase currency. Usage: Adjust/AppsFlyer.
         * */
        fun withCurrency(currency: String?) = apply { this.currency = currency }

        /**
         * @param sku Stock keeping unit id. Usage for Adjust.
         * */
        fun withSku(sku: String?) = apply { this.sku = sku }

        /**
         * @param orderId product purchased unique order id for the transaction (returned from Google API when the purchase is completed). Usage: Adjust.
         * */
        fun withOrderId(orderId: String?) = apply { this.orderId = orderId }

        /**
         * @param developerPayload Product purchased developer payload (returned from Google API when the purchase is completed). Usage: Adjust.
         * */
        fun withDeveloperPayload(developerPayload: String?) =
            apply { this.developerPayload = developerPayload }

        /**
         * @param purchaseToken product purchased token (returned from Google API when the purchase is completed). Usage for Adjust.
         * */

        fun withPurchaseToken(purchaseToken: String?) = apply { this.purchaseToken = purchaseToken }

        /**
         * @param purchaseTimestamp product purchased timestamp (returned from Google API when the purchase is completed). Usage: Adjust.
         * */
        fun withPurchaseTimestamp(purchaseTimestamp: Long) =
            apply { this.purchaseTimestamp = purchaseTimestamp }

        /**
         * @param additionalParameters additional parameters of the purchase event. Usage: Adjust/AppsFlyer.
         * */
        fun withAdditionalParams(additionalParameters: Map<String, String>) =
            apply { this.additionalParameters = additionalParameters }

        fun build() = InAppPurchase(
            type = type,
            publicKey = publicKey,
            signature = signature,
            purchaseData = purchaseData,
            developerPayload = developerPayload,
            price = price,
            currency = currency,
            sku = sku,
            orderId = orderId,
            purchaseToken = purchaseToken,
            purchaseTimestamp = purchaseTimestamp,
            additionalParameters = additionalParameters + getMainParameters()
        )

        private fun getMainParameters(): Map<String, String> =
            mapOf(
                "apd_type" to type.name,
                "apd_public_key" to publicKey.takeIf { !it.isNullOrEmpty() },
                "apd_signature" to signature.takeIf { !it.isNullOrEmpty() },
                "apd_purchase_data" to purchaseData.takeIf { !it.isNullOrEmpty() },
                "apd_developer_payload" to developerPayload.takeIf { !it.isNullOrEmpty() },
                "apd_price" to price.takeIf { !it.isNullOrEmpty() },
                "apd_currency" to currency.takeIf { !it.isNullOrEmpty() },
                "apd_sku" to sku.takeIf { !it.isNullOrEmpty() },
                "apd_order_id" to orderId.takeIf { !it.isNullOrEmpty() },
                "apd_purchase_token" to purchaseToken.takeIf { !it.isNullOrEmpty() },
                "apd_purchase_timestamp" to purchaseTimestamp.toString()
            ).mapNotNull { (key, value) -> value?.let { key to it } }.toMap()
    }

    companion object {
        /**
         * Create a new In-app Builder
         * */
        @JvmStatic
        fun newBuilder(type: Type): Builder = Builder(type)

        /**
         * Create a new In-app Builder for purchase type @see [type]
         * */
        @JvmStatic
        fun newInAppBuilder(): Builder = newBuilder(Type.InApp)

        /**
         * Create a new In-app Builder for subscription type @see [type]
         * */
        @JvmStatic
        fun newSubscriptionBuilder(): Builder = newBuilder(Type.Subs)
    }
}
