package com.appodeal.ads.initializing

/**
 * List of possible initialization errors
 */
sealed class ApdInitializationError : Throwable() {

    /**
     * Critical errors. Initialization failed.
     */
    sealed class Critical(val description: String) : ApdInitializationError() {
        /**
         * Appodeal SDK supports Android version >= 21
         */
        object SdkVersionIsNotSupported :
            Critical("Minimum supported Android api version is 21. Appodeal SDK is disabled")

        /**
         * Appodeal App Key not provided
         */
        class AppKeyIsNullOrEmpty(bundleId: String) :
            Critical(
                "The Appodeal APP KEY passed to Appodeal initialization method is null or empty. \nYou may check https://app.appodeal.com/apps to find the corresponding App Key for $bundleId"
            )
    }

    sealed class InternalError : ApdInitializationError() {
        /**
         * Internal configuration error occurred.
         *
         * Services disabled for the current session (Adjust, Appsflyer, Firebase, Facebook, etc.):
         * revenue tracking, purchase validation are not available.
         */
        class SdkConfigurationError(override val cause: Throwable? = null) : InternalError()
    }

    /**
     * Non critical errors.
     * Appodeal initialized with some errors, but able to continue working.
     */
    sealed class NonCritical(
        val componentName: String,
        val description: String?
    ) : ApdInitializationError() {

        class OtherError(componentName: String, description: String?) :
            NonCritical(componentName, description)

        /**
         * Adjust event tracking service was not initialized
         * @see <a href="https://docs.appodeal.com/">Adjust integration</a>
         */
        sealed class AdjustNotInitialized(description: String) : NonCritical(
            componentName = "Adjust",
            description = description
        ) {
            /**
             * Error occurring due to missing Adjust app token.
             *
             * You may find it visiting:
             *
             * @see <a href="https://help.adjust.com/en/article/app-settings#view-your-app-token">Adjust setup</a>
             * */
            object NoAppToken :
                AdjustNotInitialized(
                    "App token not provided, see https://help.adjust.com/en/article/app-settings#view-your-app-token"
                )

            /**
             * Error that occurs due to the lack of Adjust environment.
             *
             * Contact the support team.
             * */
            object NoEnvironment : AdjustNotInitialized("Environment not provided")
        }

        /**
         * Appsflyer event tracking service was not initialized
         * @see <a href="https://docs.appodeal.com/">Appsflyer integration</a>
         */
        sealed class AppsflyerNotInitialized(description: String) : NonCritical(
            componentName = "AppsFlyer",
            description = description
        ) {
            /**
             * Error occurring due to missing Appsflyer dev key.
             *
             * @see <a href="https://support.appsflyer.com/hc/en-us/articles/207032066-Basic-SDK-integration-guide#retrieving-the-dev-key">AppsFlyer setup</a>
             * */
            object NoDevKeyProvided :
                AppsflyerNotInitialized(
                    "DevKey not provided, see https://support.appsflyer.com/hc/en-us/articles/207032066-Basic-SDK-integration-guide#retrieving-the-dev-key"
                )

            /**
             * Error occurring due to missing Appsflyer app id.
             *
             * */
            object NoAppIdProvided : AppsflyerNotInitialized("AppId not provided")
        }

        /**
         * Facebook event tracking service was not initialized
         * @see <a href="https://docs.appodeal.com/">Facebook integration</a>
         */
        sealed class FacebookAnalyticsNotInitialized(description: String) : NonCritical(
            componentName = "Facebook Analytics",
            description = description
        ) {
            /**
             * Error that occurs due Facebook app id was not overridden.
             *
             * @see <a href="https://developers.facebook.com/docs/android/getting-started/#add-app_id">Facebook setup</a>
             * */
            object AppIdIsNotOverridden :
                FacebookAnalyticsNotInitialized(
                    "R.string.facebook_app_id string resource not overridden, see https://developers.facebook.com/docs/android/getting-started/#add-app_id"
                )

            /**
             * Error that occurs due Facebook app id was not overridden.
             *
             * @see <a href="https://developers.facebook.com/docs/android/getting-started/#client-token">Facebook setup</a>
             * */
            object ClientTokenIsNotOverridden :
                FacebookAnalyticsNotInitialized(
                    "R.facebook_client_token string resource not overridden, see https://developers.facebook.com/docs/android/getting-started/#client-token"
                )
        }

        /**
         * Firebase event tracking service was not initialized
         * @see <a href="https://docs.appodeal.com/">Firebase integration</a>
         */
        sealed class FirebaseAnalyticsNotInitialized(description: String) : NonCritical(
            componentName = "Firebase Analytics",
            description = description
        ) {
            /**
             * Error that occurs due Firebase no default options were found.
             *
             * @see <a href="https://firebase.google.com/docs/android/setup">Firebase setup</a>
             * */
            object GoogleServicesNotApplied :
                FirebaseAnalyticsNotInitialized(
                    "com.google.gms:google-services was not applied, see https://firebase.google.com/docs/android/setup"
                )
        }
    }
}
