package com.appodeal.ads.service

/**
 * An error caused during the operation of the service.
 *
 * @property componentName name ot a service.
 * @property description description of an error.
 * */
sealed class ServiceError(
    val componentName: String = String(),
    val description: String = String()
) : Throwable(message = description) {

    /**
     * Unexpected error related to the internal logic of the service.
     * */
    object Internal : ServiceError()

    /**
     * Error occurring due to initialization timeout.
     * */
    object InitializationTimeout : ServiceError()

    /**
     * Error occurring due to in-app validation timeout.
     * */
    object IAPValidationTimeout : ServiceError()

    /**
     * Error that occurs due to the unavailability of the validation service.
     * */
    object PurchaseValidationServiceNotFound : ServiceError()

    /**
     * Error that occurs if  validation service disabled
     * */
    class PurchaseValidationServiceDisabled(message: String) :
        ServiceError(message)

    sealed class Adjust(description: String) : ServiceError(
        componentName = "Adjust",
        description = description
    ) {
        /**
         * Error occurring due to missing Adjust app token.
         *
         * You may find it visiting:
         *
         * @see <a href="https://help.adjust.com/en/article/app-settings#view-your-app-token">Adjust setup</a>
         * */
        object NoAppToken :
            Adjust("App token not provided, see https://help.adjust.com/en/article/app-settings#view-your-app-token")

        /**
         * Error that occurs due to the lack of Adjust environment.
         *
         * Contact the support team.
         * */
        object NoEnvironment : Adjust("Environment not provided")

        /**
         * Error during tracking a purchase in Adjust.
         * */
        object InAppPurchaseTrackError : Adjust("Purchase track error")

        /**
         * Error during tracking a subscription in Adjust.
         * */
        object SubscriptionTrackError : Adjust("Subscription track error")

        /**
         * Error during validating an in-app purchase in Adjust.
         * */
        class PurchaseValidationError(description: String) : Adjust(description = description)
    }

    sealed class Appsflyer(description: String) : ServiceError(
        componentName = "AppsFlyer",
        description = description
    ) {
        /**
         * Error occurring due to missing Appsflyer dev key.
         *
         * @see <a href="https://support.appsflyer.com/hc/en-us/articles/207032066-Basic-SDK-integration-guide#retrieving-the-dev-key">AppsFlyer setup</a>
         * */
        object NoDevKeyProvided :
            Appsflyer(
                "DevKey not provided, see https://support.appsflyer.com/hc/en-us/articles/207032066-Basic-SDK-integration-guide#retrieving-the-dev-key"
            )

        /**
         * Error occurring due to missing Appsflyer app id.
         *
         * */
        object NoAppIdProvided : Appsflyer("AppId not provided")

        /**
         * Error during validating an in-app purchase in Appsflyer.
         * */
        class PurchaseValidationError(description: String?) :
            Appsflyer(description = description.orEmpty())

        /**
         * Error during conversion data receiving in Appsflyer.
         * */
        class ConversionDataError(description: String) :
            Appsflyer(description = description)

        /**
         * Error during attribution id receiving in Appsflyer.
         * */
        class AttributionError(description: String) :
            Appsflyer(description = description)

        /**
         * Error during initialize in Appsflyer.
         * */
        class InitializeError(description: String) :
            Appsflyer(description = description)
    }

    sealed class FacebookAnalytics(description: String) : ServiceError(
        componentName = "Facebook Analytics",
        description = description
    ) {
        /**
         * Error that occurs due Facebook app id was not overridden.
         *
         * @see <a href="https://developers.facebook.com/docs/android/getting-started/#add-app_id">Facebook setup</a>
         * */
        object AppIdIsNotOverridden :
            FacebookAnalytics(
                "R.string.facebook_app_id string resource not overridden, see https://developers.facebook.com/docs/android/getting-started/#add-app_id"
            )

        /**
         * Error that occurs due Facebook app id was not overridden.
         *
         * @see <a href="https://developers.facebook.com/docs/android/getting-started/#client-token">Facebook setup</a>
         * */
        object ClientTokenIsNotOverridden :
            FacebookAnalytics(
                "R.facebook_client_token string resource not overridden, see https://developers.facebook.com/docs/android/getting-started/#client-token"
            )
    }

    sealed class FirebaseAnalytics(description: String) : ServiceError(
        componentName = "Firebase Analytics",
        description = description
    ) {
        /**
         * Error that occurs due Firebase no default options were found.
         *
         * @see <a href="https://firebase.google.com/docs/android/setup">Firebase setup</a>
         * */
        object GoogleServicesNotApplied :
            FacebookAnalytics(
                "com.google.gms:google-services was not applied, see https://firebase.google.com/docs/android/setup"
            )
    }

    sealed class SentryAnalytics(description: String) : ServiceError(
        componentName = "Sentry Analytics",
        description = description
    ) {
        /**
         * Error occurring due to missing Sentry DSN.
         *
         * */
        object NoDsnProvided : SentryAnalytics("DSN not provided")

        /**
         * Error occurring due to missing Sentry environment.
         *
         * */
        object NoEnvironmentProvided : SentryAnalytics("Environment not provided")
    }
}
