package io.bidmachine.schema.analytics

import com.github.plokhotnyuk.jsoniter_scala.core.JsonValueCodec
import com.github.plokhotnyuk.jsoniter_scala.macros.{CodecMakerConfig, JsonCodecMaker}
import io.bidmachine.schema.analytics.meta.EventMeta
import io.bidmachine.schema.rtb.{Bid, Request}

import java.time.Instant

// Single event representing the entire auction
// For the next version of this event we might consider https://docs.metamarkets.com/docs/integration-guide-for-exchanges
case class AuctionEvent(
  timestamp: Instant,
  meta: EventMeta,
  seller: Seller, // same as in BidEvent
  request: Request, // same as in BidEvent
  
  blockedRequests: List[DSPRequest], // requests blocked by auction e.g. due to SRO block
  noBidRequests: List[DSPRequest], // including requests without bids
  bidders: List[DSPRequest], // including requests with loss bids and invalid bids
  winner: Option[DSPRequest], // in case of no winner, None
  
  rejectedAdRequestReason: Option[String] // in case request was dropped on seller level
)

object AuctionEvent {
  implicit val auctionEventCodec: JsonValueCodec[AuctionEvent] = JsonCodecMaker.make[AuctionEvent](CodecMakerConfig)

  def rejected(timestamp: Instant, meta: EventMeta, seller: Seller, request: Request, reason: String): AuctionEvent = 
    AuctionEvent(
      timestamp = timestamp,
      meta = meta,
      seller = seller,
      request = request,
      blockedRequests = Nil,
      noBidRequests = Nil,
      bidders = Nil,
      winner = None,
      rejectedAdRequestReason = Some(reason)
    )
}

case class DSPRequest(
  buyer: Buyer,

  ext: DSPRequest.Ext,
  meta: EventMeta, // only overridden labels on DSP level i.e. bidder, bid, win, response level
  
  bid: Option[Bid], // same as in BidEvent
  rejectedReason: Option[String] // in case request was dropped by Auction e.g. due to SRO block or was rejected by DSP e.g. timeout
)

object DSPRequest {

  //@todo, later add model/fields which could be overridden e.g. floors, fee
  case class Ext(
    adt: Option[String], // fills only if different from the original request adt
    maxRound: Option[Int],
    bidRound: Option[Int],
    calloutPolicy: Option[String],
    bidFloor: Option[Double],
  )

  object Ext {
    implicit val dspRequestExtCodec: JsonValueCodec[Ext] = JsonCodecMaker.make[Ext](CodecMakerConfig)

    def make(adRequest: Request): Ext =
      Ext(
        adt = adRequest.ext.adt,
        maxRound = adRequest.ext.calloutMaxRound,
        bidRound = adRequest.ext.calloutBidRound,
        calloutPolicy = adRequest.ext.calloutPolicy,
        bidFloor = Some(adRequest.ext.flr)
      )

    def make(adRequest: Request, bidRequest: Request): Ext = 
      Ext(
        adt = if (adRequest.ext.adt != bidRequest.ext.adt) bidRequest.ext.adt else None,
        maxRound = bidRequest.ext.calloutMaxRound,
        bidRound = bidRequest.ext.calloutBidRound,
        calloutPolicy = bidRequest.ext.calloutPolicy,
        bidFloor = Some(bidRequest.ext.flr)
      )
  }

  implicit val dspRequestCodec: JsonValueCodec[DSPRequest] = JsonCodecMaker.make[DSPRequest](CodecMakerConfig)

  def bidder(buyer: Buyer, ext: DSPRequest.Ext, bid: Bid, meta: EventMeta): DSPRequest =
    DSPRequest(buyer, ext, meta, Some(bid), None)

  def rejectedRequest(buyer: Buyer, ext: DSPRequest.Ext, meta: EventMeta, rejectedReason: String): DSPRequest =
    DSPRequest(buyer, ext, meta, None, Some(rejectedReason))

  def noBid(buyer: Buyer, ext: DSPRequest.Ext, meta: EventMeta): DSPRequest =
    DSPRequest(buyer, ext, meta, None, None)
}
