package io.bidmachine.schema.analytics

import java.time.Instant

import com.github.plokhotnyuk.jsoniter_scala.core.JsonValueCodec
import com.github.plokhotnyuk.jsoniter_scala.macros.{CodecMakerConfig, JsonCodecMaker}
import io.bidmachine.schema.adcom.{ConnectionType, DeviceType, Geo}
import io.bidmachine.schema.analytics.InitRequestEvent.{AndroidPlatformData, IOSPlatformData}
import io.bidmachine.schema.analytics.meta.EventMeta

/**
 * Analytics event representing the initialization of the Exchange SDK request.
 *
 * @param timestamp
 *   Event occurrence timestamp
 * @param id
 *   Unique request identifier (typically UUID)
 * @param sourceId
 *   Source identifier for the request
 * @param bundle
 *   App bundle identifier
 * @param os
 *   Operating system name
 * @param osv
 *   Operating system version
 * @param sdk
 *   SDK name
 * @param sdkver
 *   SDK version
 * @param deviceType
 *   Device type classification
 * @param connectionType
 *   Network connection type
 * @param geo
 *   Geographic location information
 * @param appVersion
 *   Application version
 * @param ifa
 *   Identifier for advertisers
 * @param ifv
 *   Identifier for vendors
 * @param bmIfv
 *   Exchange identifier for vendors
 * @param sessionId
 *   Session identifier
 * @param ip
 *   IP address
 * @param domain
 *   Domain name
 * @param displayManager
 *   Display manager configurations
 * @param meta
 *   Event metadata
 * @param make
 *   Device manufacturer
 * @param model
 *   Device model
 * @param hwv
 *   Hardware version
 * @param initialDC
 *   Initial data center
 * @param routedDC
 *   Routed data center
 * @param installTime
 *   App installation timestamp
 * @param firstLaunchTime
 *   First app launch timestamp
 * @param lastSessionDuration
 *   Duration of the last session in milliseconds
 * @param androidMinApiLevel
 *   Minimum Android API level required
 * @param androidKotlinVersion
 *   Kotlin version used (Android)
 * @param iosMinOsVersion
 *   Minimum iOS version required
 * @param iosOsExecutionEnv
 *   iOS execution environment
 * @param storecat
 *   App category name in app store
 * @param storesubcat
 *   App subcategories name in app store
 * @param fmwname
 *   Framework name. (unknown, native, unity)
 * @param installReferrerUrl
 *   The referrer URL of the installed package. Depends on install referer library.
 * @param installReferrerClickTimestamp
 *   The client-side timestamp, when the referrer click happened. Depends on install referer library.
 * @param installReferrerClickServerTimestamp
 *   The server-side timestamp, when the referrer click happened. Depends on install referer library.
 * @param installBeginServerTimestamp
 *   The server-side timestamp, when app installation began. Depends on install referer library.
 * @param installVersion
 *   The app's version at the time when the app was first installed. Depends on install referer library.
 * @param isGooglePlayInstant
 *   Indicates whether your app's instant experience was launched within the past 7 days. Depends on install referer
 *   library.
 * @param sdkInstallTime
 *   The first recorded SDK initialization timestamp, marking the first time the SDK was present in this specific app.
 *   This field is empty if the SDK was updated from a legacy version that did not support it.
 * @param androidPlatformData
 *   Android specific data
 * @param iosPlatformData
 *   iOS specific data
 */
case class InitRequestEvent(
  timestamp: Instant,
  id: String, // Unique Request Id (possible UUID)
  sourceId: String,
  bundle: String,
  os: Option[String],
  osv: String,
  sdk: String,
  sdkver: String,
  deviceType: Option[DeviceType],
  connectionType: Option[ConnectionType],
  geo: Option[Geo],
  appVersion: Option[String],
  ifa: Option[String],
  ifv: Option[String],
  bmIfv: Option[String],
  sessionId: Option[String],
  ip: Option[String],
  domain: String,
  displayManager: Option[List[InitDisplayManager]],
  meta: EventMeta,
  make: Option[String],
  model: Option[String],
  hwv: Option[String],
  initialDC: Option[String],
  routedDC: Option[String],
  installTime: Option[Instant],
  firstLaunchTime: Option[Instant],
  lastSessionDuration: Option[Long],
  androidMinApiLevel: Option[Int],
  androidKotlinVersion: Option[String],
  iosMinOsVersion: Option[String],
  iosOsExecutionEnv: Option[String],
  storecat: Option[String],
  storesubcat: Option[List[String]],
  fmwname: Option[String],
  installReferrerUrl: Option[String],
  installReferrerClickTimestamp: Option[Instant],
  installReferrerClickServerTimestamp: Option[Instant],
  installBeginServerTimestamp: Option[Instant],
  installVersion: Option[String],
  isGooglePlayInstant: Option[Boolean],
  sdkInstallTime: Option[Instant],
  androidPlatformData: Option[AndroidPlatformData],
  iosPlatformData: Option[IOSPlatformData]
)

object InitRequestEvent {

  /**
   * @param startInitTime
   *   Timestamp when the BidMachineInitProvider instance was created
   * @param initDuration
   *   Duration of the BidMachineInitProvider initialization in milliseconds
   * @param initCompleted
   *   Indicates whether initialization completed successfully. A value of false may indicate an unexpected process
   *   termination (e.g., ANR or crash).
   */
  case class ContentProviderStartupMetrics(
    startInitTime: Option[Instant],
    initDuration: Option[Long],
    initCompleted: Option[Boolean]
  )

  /**
   * @param minApiLevel
   *   Min supported Android API version
   * @param kotlinVersion
   *   Kotlin runtime version
   * @param contentProviderStartupMetrics
   *   Statistics related to BidMachineInitProvider startup metrics
   */
  case class AndroidPlatformData(
    minApiLevel: Option[Int],
    kotlinVersion: Option[String],
    contentProviderStartupMetrics: Option[List[ContentProviderStartupMetrics]]
  )

  /**
   * @param minOsVersion
   *   Min supported iOS OS version
   * @param osExecutionEnv
   *   An OSExecutionEnvironment value indicating iOS device execution environment;
   *   https://github.com/bidmachine/proto/blob/242514849a3084f02647dcc53a7e9f5eac2ee75c/bidmachine/protobuf/init.proto#L24
   */
  case class IOSPlatformData(
    minOsVersion: Option[String],
    osExecutionEnv: Option[Int]
  )

  implicit val contentProviderStartupMetricsCodec: JsonValueCodec[ContentProviderStartupMetrics] =
    JsonCodecMaker.make[ContentProviderStartupMetrics](CodecMakerConfig)
  implicit val androidPlatformDataCodec: JsonValueCodec[AndroidPlatformData]                     =
    JsonCodecMaker.make[AndroidPlatformData](CodecMakerConfig)
  implicit val iOSPlatformDataCodec: JsonValueCodec[IOSPlatformData]                             =
    JsonCodecMaker.make[IOSPlatformData](CodecMakerConfig)
  implicit val initSessionEventCodec: JsonValueCodec[InitRequestEvent]                           =
    JsonCodecMaker.make[InitRequestEvent](CodecMakerConfig)
}
